#include "./shape.hpp"

bool Shape::sampleSurface(
	const Point& refPosition,
	const Vec3& refNormal,
	double u1,
	double u2,
	double u3,
	Point& outPosition,
	Vec3& outNormal,
	double& outPDF) const
{
	(void)refPosition;
	(void)refNormal;
	(void)u1;
	(void)u2;
	(void)u3;
	(void)outPosition;
	(void)outNormal;
	(void)outPDF;
	outPDF = 0.0f;
	return false;
}

double Shape::pdfSA(
	const Point& refPosition,
	const Vec3& refNormal,
	const Point& surfPosition,
	const Vec3& surfNormal) const
{
	(void)refNormal;
	Vec3 incoming = surfPosition - refPosition;
	double dist = incoming.normalize();
	return squared(dist) * surfaceAreaPDF() / std::fabs(dot(surfNormal, incoming));
}


ShapeSet::ShapeSet(){
	
}

ShapeSet::~ShapeSet(){
	
}

void ShapeSet::addShape(Shape* shape){
	shapes.push_back(shape);
}

void ShapeSet::findLights(std::list<Shape*>& outLights)
{
	
	for (std::vector<Shape*>::iterator iter = shapes.begin();
		iter != shapes.end();
		iter++)
	{
		Shape *pShape = *iter;
		if(pShape->isLight()){
			outLights.push_back(pShape);
		}
	}
}


bool ShapeSet::intersect(Intersection& intersection){
	bool doesIntersect = false;
	
	for(auto it = shapes.begin(); it != shapes.end(); ++it){
		Shape* curShape = *it;
		
		if(curShape->intersect(intersection) == true){
			doesIntersect = true;
			//break can'T go here. All must be tested because of depth
		}
	}
	
	return doesIntersect;
}

bool ShapeSet::doesIntersect(const Ray& ray){
	bool doesIntersect = false;
	
	for(auto it = shapes.begin(); it != shapes.end(); ++it){
		Shape* curShape = *it;
		
		if(curShape->doesIntersect(ray) == true){
			doesIntersect = true;
			break;
		}
	}
	
	return doesIntersect;
}

//PLANE

Plane::Plane(const Point& _position, const Vec3& _normal, const Color& _color): position(_position), color(_color){
	Vec3 d(_normal);
	normal = d.normalized();
}

Plane::~Plane(){
	
}

bool Plane::intersect(Intersection& intersection){
	double dDotN = dot(intersection.ray.direction,normal);
	
	
	if(dDotN == 0.0f){
		//The ray is either fully inside the plane or parallel and outside.
		return false;
	}
	
	//Point of intersection
	double t = dot(position- intersection.ray.origin, normal) / dDotN;
	
	if(t <= RAY_T_MIN || t >= intersection.t){
		return false; //Either too close or too far.
	}
	
	intersection.normal = normal;
	intersection.t = t;
	intersection.pShape = this;
	intersection.color = color;
	
	return true;
	
} //Goes and finds where the shape intersects.

bool Plane::doesIntersect(const Ray& ray){
	
	//Check intersection
	double dDotN = dot(ray.direction,normal);
	
	if(dDotN == 0.0f){
		//The ray is either fully inside the plane or parallel and outside.
		return false;
	}
	
	//Point of intersection
	double t = dot(position- ray.origin, normal) / dDotN;
	
	if(t <= RAY_T_MIN || t >= ray.tMax){
		return false; //Either too close or too far.
	}
	return true;
	
	
} //Checks if the ray hits the shape or not.

//SPHERE

Sphere::Sphere(const Point& _center, double _radius, const Color& _color): center(_center), radius(_radius), color(_color){
	
}

Sphere::~Sphere(){
	
}

bool Sphere::intersect(Intersection& intersection){
	Ray localRay = intersection.ray;
	localRay.origin -= center; //Offset it to center it.
	
	//Quadratic cofficients.
	double a = localRay.direction.length_squared();
	double b = 2*dot(localRay.direction,localRay.origin);
	double c = localRay.origin.length_squared() - square(radius);
	
	double disc = square(b)-4*a*c;
	
	//0 times intersected
	if(disc < 0.0){
		return false;
	}
	
	double t1 = (-b - std::sqrt(disc)) / (2.0*a);
	double t2 = (-b + std::sqrt(disc)) / (2.0*a);
	
	if(t1 > RAY_T_MIN && t1 < intersection.t) {
		intersection.t = t1;
	}
	else if(t2 > RAY_T_MIN && t2 < intersection.t) {
		intersection.t = t2;
	}
	 else { //Neither hit precisely enough.
		return false;
	}
	
	//intersection.normal = localRay.calculate(t) = (intersection.ray.origin - center) + t * intersection.ray.direction;
	//intersection.normal = localRay.calculate(intersection.t).normalized();
	Point hitPoint = intersection.ray.calculate(intersection.t);
	intersection.normal = (hitPoint - center).normalized();
	
	intersection.pShape = this;
	intersection.color = color;
	return true;
	
} //Goes and finds where the shape intersects.

bool Sphere::doesIntersect(const Ray& ray){
	Ray localRay = ray;
	localRay.origin -= center; //Offset it to center it.
	
	//Quadratic cofficients.
	double a = localRay.direction.length_squared();
	double b = 2*dot(localRay.direction,localRay.origin);
	double c = localRay.origin.length_squared() - square(radius);
	
	double disc = square(b)-4*a*c;
	
	//0 times intersected
	if(disc < 0.0){
		return false;
	}
	
	double t1 = (-b - std::sqrt(disc)) / (2.0*a);
	
	
	if(t1 > RAY_T_MIN && t1 < ray.tMax){
		return true;
		
	}
	
	double t2 = (-b + std::sqrt(disc)) / (2.0*a);
	if(t2 > RAY_T_MIN && t2 < ray.tMax){
		return true;
		
	}
	
	return false;
	
	
} //Checks if the ray hits the shape or not.
