#include "electronics_visual_model.hpp"

#include "exceptions/exceptions.hpp"

const ImVec2& ElectricalVisualNode::getPosition() const{
	if(component == nullptr){
		logError("Component is nullptr!");
	}
	return component->getPosition();
}

void ElectricalVisualNode::setPosition(double x, double y) {
	if(component == nullptr){
		logWarn("Component is nullptr!");
		return;
	}
	component->setPosition(x,y);
}

ElectricalVisualNode::ElectricalVisualNode(){
	rotateTo(0.0);
}

ElectricalVisualNode::~ElectricalVisualNode(){
	
}

void ElectricalVisualNode::rotateTo(double deg){
	rotation = deg;
	
	rad = rotation * M_PI / 180.0f;
	
	s = sinf(rad);
	c = cosf(rad);
	
	if(component != nullptr){
		component->setRotation(deg);
	}
}

void ElectricalVisualNode::loadTextureSelf(const char* filepath,const std::string& relImagePath){
	std::string fullPath = std::string(GRAPHICS_PATH) + relImagePath + filepath;
	
	texture = loadTexture(fullPath.c_str(),textureWidth,textureHeight);
	textureLoaded = true;
	
	imageAspectRatio = static_cast<double>(textureWidth)/ static_cast<double>(textureHeight);
}

void ElectricalVisualNode::update(double offX, double offY){
	bbox.minX = getPosition().x+offX;
	bbox.minY = getPosition().y+offY;
	bbox.maxX = getPosition().x+offX+DEFAULT_SIZE_X;
	bbox.maxY = getPosition().y+offY+DEFAULT_SIZE_Y/imageAspectRatio;
}
void ElectricalVisualNode::draw(bool show_overlay_text, ImDrawList* draw_list,double offX, double offY){
	if(textureLoaded == false || component == nullptr){
		return;
	}
	
	if(InputState::getInstance()->dragged2DElement == component){
		draw_list->AddRect(ImVec2(getPosition().x + offX, getPosition().y + offY),ImVec2(getPosition().x + DEFAULT_SIZE_X+offX, getPosition().y + DEFAULT_SIZE_Y/imageAspectRatio+offY),HIGHLIGHT_COLOR);
	}
	if(rad != 0.0){
		draw_list->AddRectFilled(ImVec2(getPosition().x + offX, getPosition().y + offY),ImVec2(getPosition().x + DEFAULT_SIZE_X+offX, getPosition().y + DEFAULT_SIZE_Y/imageAspectRatio+offY),AABB_COLOR);
	}
	
	/*draw_list->AddImage(
		(void*)(intptr_t)texture,   // OpenGL texture ID cast to void*
		ImVec2(getPosition().x + offX, getPosition().y + offY),                    // top-left
		ImVec2(getPosition().x + DEFAULT_SIZE_X+offX, getPosition().y + DEFAULT_SIZE_Y/imageAspectRatio+offY) // bottom-right
	);*/
	
	double x = getPosition().x + offX;
	double y = getPosition().y + offY;

	// width/height
	double w = DEFAULT_SIZE_X;
	double h = DEFAULT_SIZE_Y / imageAspectRatio;

	// corners (top-left origin)
	ImVec2 p0 = ImVec2(x, y);
	ImVec2 p1 = ImVec2(x + w, y);
	ImVec2 p2 = ImVec2(x + w, y + h);
	ImVec2 p3 = ImVec2(x, y + h);

	// rotate each corner around component origin (top-left)
	auto rotatePoint = [&](ImVec2 p) -> ImVec2 {
		double dx = p.x - x;
		double dy = p.y - y;
		double rx = dx * c - dy * s;
		double ry = dx * s + dy * c;
		return ImVec2(rx + x, ry + y);
	};

	p0 = rotatePoint(p0);
	p1 = rotatePoint(p1);
	p2 = rotatePoint(p2);
	p3 = rotatePoint(p3);

	draw_list->AddImageQuad(
		(void*)(intptr_t)texture,
		p0, p1, p2, p3,
		ImVec2(0,0), ImVec2(1,0), ImVec2(1,1), ImVec2(0,1)
	);

	if(show_overlay_text == true){
		ImFont* font = ImGui::GetFont();
	
		float fontSize = ImGui::GetFontSize()*OVERLAY_FONT_SIZE_SCALE;
		
		ImVec2 textPos(getPosition().x + offX, getPosition().y + DEFAULT_SIZE_Y/imageAspectRatio+offY+ OVERLAY_TEXT_OFFSET);
		draw_list->AddText(font,fontSize,textPos,OVERLAY_TEXT_COLOR, component->getStats());
	}
	
	
	
}

void ElectricalVisualNode::setComponent(double _x, double _y,ElectricalComponent* c,const std::vector<ImVec2>& offsets){
	component = c;
	c->serializedRef = nullptr;
	c->setOffsets(offsets);
	setPosition(_x,_y);
	
}


WireSegment::WireSegment(Node* _p1, Node* _p2){
	p1 = _p1;
	p2 = _p2;
}

void WireSegment::draw(ImDrawList* draw_list,double offX, double offY){
	ImVec2 offvec(p1->getPosition().x+offX,p1->getPosition().y+offY);
	ImVec2 offvec2(p2->getPosition().x+offX,p2->getPosition().y+offY);
	if(InputState::getInstance()->dragged2DElement == this){
		draw_list->AddLine(offvec, offvec2, HIGHLIGHT_COLOR, wireStrength);
	} else {
		draw_list->AddLine(offvec, offvec2, WIRE_COLOR, wireStrength);
	}
	
}

void WireSegment::update(double offX, double offY){
	double _minX=std::min(p1->getPosition().x,p2->getPosition().x);
	double _minY=std::min(p1->getPosition().y,p2->getPosition().y);
	double _maxX=std::max(p1->getPosition().x,p2->getPosition().x);
	double _maxY=std::max(p1->getPosition().y,p2->getPosition().y);
	
	bbox.minX =_minX;
	bbox.minY = _minY;
	bbox.maxX = _maxX;
	bbox.maxY = _maxY;
}

void ElectricalNodeCollection::setRelImagePath(const std::string& s){
	relImagePath = s;
}

ElectricalNodeCollection::ElectricalNodeCollection(GeneralSimulation* c){
	simulation = c;
}

ElectricalNodeCollection::ElectricalNodeCollection(){
	simulation = nullptr;
}

ElectricalNodeCollection::~ElectricalNodeCollection(){
	clear();
}

void ElectricalNodeCollection::addVisualNode(ElectricalVisualNode* node){
	visualNodes.emplace_back(node);
}

void ElectricalNodeCollection::draw(bool show_overlay_text,ImDrawList* draw_list,double offX, double offY, double width, double height,bool portHighlight){
	for(size_t i=0;i<visualNodes.size();i++){
		//if(visualNodes[i]->bbox.isRectInside(offX,offY,width,height) == true){
			visualNodes[i]->draw(show_overlay_text,draw_list,offX,offY);
			if(portHighlight == true){
				std::vector<Port*> ps = visualNodes[i]->component->getPorts();
				for(size_t j = 0;j<ps.size();j++){
					ImVec2 p = ps[j]->getPosition();
					
					draw_list->AddCircleFilled(
						ImVec2(p.x +offX+ PORT_RADIUS/2.0, p.y+offY +PORT_RADIUS/2.0 ),
						PORT_RADIUS,
						HIGHLIGHT_COLOR
					);
				}
			}
		//}
		
	}
	
	
	
	for(size_t i=0;i<wires.size();i++){
		//if(visualNodes[i]->bbox.isRectInside(offX,offY,width,height) == true){
			wires[i]->draw(draw_list,offX,offY);
		//}
	}
	
	//

}
void ElectricalNodeCollection::update(double offX, double offY){
	for(size_t i=0;i<visualNodes.size();i++){
		visualNodes[i]->update(offX,offY);
	}
	
	for(size_t i=0;i<wires.size();i++){
		wires[i]->update(offX,offY);
	}
	
	//

}

bool ElectricalNodeCollection::addWireSegment(Node* a, Node* b){
	if(a == nullptr || b == nullptr){
		
		//TODO error
		return false;
	}
	wires.push_back(new WireSegment(a,b));
	return true;
}

void ElectricalNodeCollection::connectPort(Port* a, Port* b){
	if(simulation == nullptr){
		//TODO error
	}
	
	bool success = addWireSegment(a,b);
	if(success == false){
		//TODO error
	}
	
	simulation->connectPort(a,b);
}



void ElectricalNodeCollection::solve(){
	if(simulation == nullptr){
		logError("Simulation is nullptr!");
	}
	simulation->solve();
}

void ElectricalNodeCollection::checkDrag(){

	if(InputState::getInstance()->dragMode == false){
		if(InputState::getInstance()->left_mouse_down == true){
			for(size_t i=0;i<visualNodes.size();i++){
				if(visualNodes[i]->bbox.isPointInside(InputState::getInstance()->mouse_x,InputState::getInstance()->mouse_y) == true){
					InputState::getInstance()->dragged2DElement = visualNodes[i]->component;
					InputState::getInstance()->dragging = true;
					return;
				}
			}
			
			for(size_t i=0;i<wires.size();i++){
				if(wires[i]->bbox.isPointInside(InputState::getInstance()->mouse_x,InputState::getInstance()->mouse_y) == true){
					InputState::getInstance()->dragged2DElement = wires[i];
					InputState::getInstance()->dragging = true;
					return;
				}
			}
		}
	} else {
		if(InputState::getInstance()->left_mouse_down == true){
			if(InputState::getInstance()->dragging == false){
				for(size_t i=0;i<visualNodes.size();i++){
					if(visualNodes[i]->bbox.isPointInside(InputState::getInstance()->mouse_x,InputState::getInstance()->mouse_y) == true){
						InputState::getInstance()->dragged2DElement = visualNodes[i]->component;
						InputState::getInstance()->dragging = true;
						return;
					}
				}
			}
		}
	}
	
}

void ElectricalNodeCollection::tryRotate(double deg){
	for(size_t i=0;i<visualNodes.size();i++){
		if(visualNodes[i]->bbox.isPointInside(InputState::getInstance()->mouse_x,InputState::getInstance()->mouse_y) == true){
			visualNodes[i]->rotateTo(deg);
			return;
		}
	}
}

void ElectricalNodeCollection::tryInvert(){
	for(size_t i=0;i<visualNodes.size();i++){
		if(visualNodes[i]->component->isVoltageSource() == true && visualNodes[i]->bbox.isPointInside(InputState::getInstance()->mouse_x,InputState::getInstance()->mouse_y) == true){
			double v = visualNodes[i]->component->getVoltage();
			
			if(v == 0.0){
				visualNodes[i]->component->setLastVoltage(DIGITAL_HIGH);
			} else {
				visualNodes[i]->component->setLastVoltage(DIGITAL_LOW);
			}
			visualNodes[i]->component->setSimulatedString();
			
			return;
		}
	}
}

void ElectricalNodeCollection::checkDelete(Node* c){
	
	auto* ec = dynamic_cast<WireSegment*>(c);
	
	if(ec){
		for(size_t i =0;i<wires.size();i++){
			if(wires[i] == ec){
				if(simulation !=nullptr){
					simulation->deleteConnection(wires[i]->p1,wires[i]->p2);
				}
				wires.erase(wires.begin() + i);
			}
		}
		return;
	}
	
	for(size_t i=0;i<visualNodes.size();i++){
		if(visualNodes[i]->component == c){
			std::vector<Port*> ps = visualNodes[i]->component->getPorts();
			for(size_t i =0;i<ps.size();i++){
				for (size_t j = wires.size(); j-- > 0;) {
					if(wires[j]->p1 == ps[i] || wires[j]->p2 == ps[i]){
						wires.erase(wires.begin() + j);
					}
				}
			}
			
			Node* target = visualNodes[i]->component;
			visualNodes[i]->component = nullptr;
			
			simulation->clearElement(target);
			
			ElectricalVisualNode* node = visualNodes[i];
			delete node;
			node = nullptr;
			
			visualNodes.erase(visualNodes.begin()+i);
			break;
		}
	}
}

void ElectricalNodeCollection::clear(){
	wires.clear();
	visualNodes.clear();
	if(simulation != nullptr){
		simulation->clear();
	}
	
}
