#ifndef ELECTRICAL_MODEL_DATA_HPP
#define ELECTRICAL_MODEL_DATA_HPP

#include <memory>
#include <unordered_set>
#include <vector>
#include <iostream>
#include <cmath>
//#include <glm/glm.hpp>

#include "world_pieces/electronics/base_component.hpp"
#include "world_pieces/electronics/port.hpp"
#include "world_pieces/electronics/net.hpp"
#include "world_simulators/circuit_simulation.hpp"
#include "world_simulators/general_simulation.hpp"
#include "world_simulators/digital_simulation.hpp"

#include "world_pieces/electronics/null_port.hpp"
#include "world_pieces/electronics/voltage_source.hpp"
#include "world_pieces/electronics/current_source.hpp"
#include "world_pieces/electronics/resistor.hpp"
#include "world_pieces/electronics/diode.hpp"
#include "world_pieces/electronics/capacitor.hpp"
#include "world_pieces/electronics/op_amp.hpp"
#include "world_pieces/electronics/npn_transistor.hpp"
#include "world_pieces/electronics/logical_pin.hpp"
#include "world_pieces/electronics/GND.hpp"
#include "world_pieces/electronics/node.hpp"

#include "world_pieces/digital/and_gate.hpp"
#include "world_pieces/digital/or_gate.hpp"
#include "world_pieces/digital/output_bit.hpp"
#include "world_pieces/digital/input_bit.hpp"
#include "world_pieces/digital/xor_gate.hpp"
#include "world_pieces/digital/nand_gate.hpp"
#include "world_pieces/digital/invert_gate.hpp"

#include "imgui.h"
#include "imgui_impl_glfw.h"
#include "imgui_impl_opengl3.h"
#include <stdio.h>
#define GL_SILENCE_DEPRECATION
#if defined(IMGUI_IMPL_OPENGL_ES2)
#include <GLES2/gl2.h>
#endif
#include <GLFW/glfw3.h> // Will drag system OpenGL headers

#include "utils.hpp"
#include "input_state.hpp"

#include "GLOBAL_CONSTANTS.hpp"

#include "world_pieces/abstract/AABB.hpp"

class ElectricalVisualNode{
	private:
		bool textureLoaded = false;
		double imageAspectRatio = 1.0;
		
	protected:
		
		
		int textureWidth;
		int textureHeight;
		GLuint texture;
		
		double rotation=0.0;
		
		bool invalid = false;
		
		
		double s = 0.0;
		double c = 0.0;
		
		double rad = 0.0;
		
	public:
		AABB bbox;
	
		ElectricalComponent* component = nullptr;
		const ImVec2& getPosition() const;
		
		void setPosition(double x, double y);
		
		ElectricalVisualNode();
		
		virtual ~ElectricalVisualNode();
		
		void rotateTo(double deg);
		
		void loadTextureSelf(const char* filepath,const std::string& relImagePath);
		
		void update(double offX, double offY);
		void draw(bool show_overlay_text, ImDrawList* draw_list,double offX, double offY);
		
		void setComponent(double _x, double _y,ElectricalComponent* c,const std::vector<ImVec2>& offsets);
};

class WireSegment: public Node{
	private:
		
		
		float wireStrength = 2.0f;
	public:
		AABB bbox;
		
		Node* p1;
		Node* p2;
		
		WireSegment(Node* _p1, Node* _p2);
		
		void draw(ImDrawList* draw_list,double offX, double offY);
		
		void update(double offX, double offY);
};

class ElectricalNodeCollection{
	private:
		std::vector<ElectricalVisualNode*> visualNodes;
		std::vector<WireSegment*> wires;
		
		std::string relImagePath;
		
	public:
		void setRelImagePath(const std::string& s);
	
		GeneralSimulation* simulation = nullptr;
	
		ElectricalNodeCollection(GeneralSimulation* c);
		ElectricalNodeCollection();
		
		~ElectricalNodeCollection();
		
		void addVisualNode(ElectricalVisualNode* node);
		
		void draw(bool show_overlay_text,ImDrawList* draw_list,double offX, double offY, double width, double height,bool portHighlight);
		
		void update(double offX, double offY);
		
		template<typename T, typename... Args>
		T& addComponent(double x, double y,const char* filepath,const std::vector<ImVec2>& offsets, Args&&... args);
		
		bool addWireSegment(Node* a, Node* b);
		
		void connectPort(Port* a, Port* b);
		
		
		
		void solve();
		
		void checkDrag();
		
		void tryRotate(double deg);
		
		void tryInvert();
		
		void checkDelete(Node* c);
		
		void clear();
};

template<typename T, typename... Args>
T& ElectricalNodeCollection::addComponent(double x, double y,const char* filepath,const std::vector<ImVec2>& offsets, Args&&... args) {
	
	
	if(simulation == nullptr){
		//TODO error
	}
	
	T& ref = simulation->addComponent<T>(std::forward<Args>(args)...);
	
	ElectricalVisualNode* node = new ElectricalVisualNode();
	node->setComponent(x,y,&ref,offsets);
	node->loadTextureSelf(filepath,relImagePath);
	addVisualNode(node);
	
	return ref;
}

#endif