#include "electronic_module_saving.hpp"

void to_json(json& j, const ImVec2& v) {
    j = json::array({ v.x, v.y });
}

void from_json(const json& j, ImVec2& v) {
    v.x = j.at(0).get<float>();
    v.y = j.at(1).get<float>();
}

void to_json(json& j, const PortConnection& c) {
    j = json{
        {"portIndex", c.portIndex},
        {"otherComponentId", c.otherComponentId},
        {"otherPortIndex", c.otherPortIndex}
    };
}

void from_json(const json& j, PortConnection& c) {
    j.at("portIndex").get_to(c.portIndex);
    j.at("otherComponentId").get_to(c.otherComponentId);
    j.at("otherPortIndex").get_to(c.otherPortIndex);
}

void to_json(json& j, const CustomElectricalComponent& d) {
	j = json::object();
	j["type"]     = d.type;
	j["position"] = { d.position.x, d.position.y };
	j["texture"]  = d.texture;
	j["pins"]     = d.pins;
	j["rotation"] = d.rotation;
    j["floats"] = d.floats;
    j["connections"] = d.connections;
    j["id"] = d.id;
	
}

void from_json(const json& j, CustomElectricalComponent& d) {
    j.at("type").get_to(d.type);

    auto p = j.at("position");
    d.position = {p[0], p[1]};

    j.at("texture").get_to(d.texture);
    j.at("pins").get_to(d.pins);
	
	j.at("rotation").get_to(d.rotation);
	
	

    if (j.contains("floats")){
        j.at("floats").get_to(d.floats);
	}
    
    if (j.contains("connections")){
        j.at("connections").get_to(d.connections);
	}
    
    if (j.contains("id")){
        j.at("id").get_to(d.id);
	}
}

ElectricalComponentCollection::ElectricalComponentCollection(){
	
}

void ElectricalComponentCollection::loadDefaultsElectrical(){
	std::string path = std::string(ELECTRONICS_COMPONENTS_DIRECTORY);
	for (const auto & entry : fs::directory_iterator(path)){
		this->loadFrom(entry.path(),entry.path().stem());
	}
}
void ElectricalComponentCollection::loadDefaultsDigital(){
	std::string path = std::string(DIGITAL_COMPONENTS_DIRECTORY);
	for (const auto & entry : fs::directory_iterator(path)){
		this->loadFrom(entry.path(),entry.path().stem());
	}
}

void ElectricalComponentCollection::loadFrom(const std::string& filename,const std::string& minFileName){ //Loads and adds to the file.
	std::vector<CustomElectricalComponent> temp = loader.loadComponents(filename);
	
	
	if (!temp.empty()) {
		temp[0].importedName = minFileName;
		
		CustomElectricalComponent* c = new CustomElectricalComponent(temp[0]);
        components.push_back(c);
	} else {
		std::cerr << "Warning: loader returned empty vector for " << filename << "\n";
		return;
	}
}

void ElectricalComponentCollection::loadFrom(const std::string& filename) { // Loads scene file with all components
	std::vector<CustomElectricalComponent> temp = loader.loadComponents(filename);
	
	// Clear existing components (if loading a scene)
	components.clear();
	
	// Load all components from the file
	for (const auto& comp : temp) {
		CustomElectricalComponent* c = new CustomElectricalComponent(comp);
		components.push_back(c);
	}
	
	if (temp.empty()) {
		std::cerr << "Warning: loader returned empty vector for " << filename << "\n";
	}
}

void ElectricalComponentCollection::seedTo(ElectricalNodeCollection& electronics) {
	
	electronics.clear();
	//Create all runtime components
	std::unordered_map<int, ElectricalComponent*> idToComponent;
	for (size_t i = 0; i < components.size(); ++i) {
		CustomElectricalComponent* c = components[i];

		// Use ID from loaded component, or assign if missing
		if (c->id < 0) {
			c->id = static_cast<int>(i);;
		}

		// Create runtime component
		c->selfComp = ComponentFactory::create(electronics, *c);

		if (c->selfComp)
			c->selfComp->setSerializedRef(c);

		idToComponent[c->id] = c->selfComp;
	}

	//Reconnect all port connections
	for (auto* c : components) {
		if (!c->selfComp) continue;

		for (const auto& conn : c->connections) {

			// Enforce canonical direction to avoid duplicates
			if (c->id > conn.otherComponentId)
				continue;

			if (c->id == conn.otherComponentId &&
				conn.portIndex >= conn.otherPortIndex)
				continue;

			// Lookup other component
			auto it = idToComponent.find(conn.otherComponentId);
			if (it == idToComponent.end()) continue;

			ElectricalComponent* otherComp = it->second;
			if (!otherComp) continue;

			// Get ports
			Port* myPort = c->selfComp->getPort(conn.portIndex);
			Port* otherPort = otherComp->getPort(conn.otherPortIndex);

			if (!myPort || !otherPort) continue;

			electronics.connectPort(myPort, otherPort);
		}
	}

	//Optionally restore net properties (e.g., set GND)
	for (auto* c : components) {
		for (Port* p : c->selfComp->ports) {
			if (p->isGroundPort) {
				if (p->net) p->net->setToGND();
			}
		}
	}
}

void ElectricalComponentCollection::save() { //ONLY for safe ones
	std::vector<CustomElectricalComponent> copy;
	copy.reserve(components.size());
	for (auto c : components) {
		copy.push_back(*c); // copy each pointed object
	}
	loader.saveComponents(fname, copy);
}

void ElectricalComponentCollection::saveTo(const std::string& fn) {
	std::vector<CustomElectricalComponent> copy;
	copy.reserve(components.size());

	// Assign stable IDs for serialization
	for (size_t i = 0; i < components.size(); ++i) {
		components[i]->id = i;
	}

	// Build a copy of each component including connections
	for (auto* c : components) {
		CustomElectricalComponent temp = *c; // copy basic data

		// Clear previous connections to rebuild
		temp.connections.clear();

		if (!c->selfComp) { //Should skip this because no valid pointer is assigned to it.
			//copy.push_back(temp);
			continue;
		}

		// Record all port connections
		std::set<std::pair<Port*, Port*>> recorded; // globally for this component

		for (Port* p : c->selfComp->ports) {
			if (!p->net) continue;

			for (Port* connectedPort : p->net->ports) {
				if (!connectedPort || connectedPort == p) continue;

				// Always order ports to avoid duplicate reverse recording
				Port* a = p < connectedPort ? p : connectedPort;
				Port* b = p < connectedPort ? connectedPort : p;

				if (recorded.count({a,b})) continue; // already recorded
				recorded.insert({a,b});

				// safe to push connection now
				int otherCompId = connectedPort->serializedRef->id;
				int otherPortIndex = connectedPort->index;

				temp.connections.push_back({p->index, otherCompId, otherPortIndex});
			}
		}

		copy.push_back(temp);
	}

	loader.saveComponents(fn, copy);
}

ElectricalComponentCollection::ElectricalComponentCollection(const std::string& filename){
	fname = filename + ECF_FILE_EXTENSION;
}

ElectricalComponentCollection::~ElectricalComponentCollection(){
	components.clear();
}

void ElectricComponentLoader::saveComponents(const std::string& filename, const std::vector<CustomElectricalComponent>& components) {
	json j = components; // vector is automatically serializable
	std::ofstream file(filename);
	if (!file) {
		std::cerr << "Error opening file for writing: " << filename << "\n";
		return;
	}
	file << j.dump(4); // pretty print with 4 spaces
}


std::vector<CustomElectricalComponent> ElectricComponentLoader::loadComponents(const std::string& filename) {
	std::ifstream file(filename);
	if (!file) {
		std::cerr << "Error opening file for reading: " << filename << "\n";
		return {};
	}

	json j;
	file >> j;

	std::vector<CustomElectricalComponent> components = j.get<std::vector<CustomElectricalComponent>>();
	return components;
}