import Mathlib

namespace CivSim

section

structure Nation where
  name                     : String
  id                       : Nat
  gdp                      : Rat
  controlled_countries     : Nat
  gdp_growth_rate           : Rat
  economic_stability        : Rat
  authoritarian_level       : Rat
  regime_stability          : Rat
  risk_tolerance            : Rat
  globalism_level           : Rat
  policy_inertia            : Rat
  military_power            : Rat
  military_growth_rate      : Rat
  security_pressure         : Rat
  war_losses                : Rat
  war_fatigue               : Rat
  resources                 : Rat
  claims                    : Nat
  supporters                : Nat
  threat_perception_bias    : Rat
  information_quality       : Rat
  local_tension             : Rat
  at_war                    : Bool
  neighbors                 : Nat
  -- Calculated fields
  war_contribution          : Rat
  gdp_max                   : Rat
  authoritarian_level_growth_rate : Rat
  economic_stability_growth_rate  : Rat
  regime_stability_growth_rate    : Rat
  risk_tolerance_growth_rate      : Rat
  war_fatigue_growth_rate          : Rat
  globalism_level_growth_rate      : Rat
  global_losses                   : Rat

  -- Positions
  x : Rat
  y : Rat

def BASE_SURRENDER_THRESHOLD : Rat := 1000.0
def BASE_GDP : Rat := 500.0

/-- Default "Austria-like" nation constructor --/
def default_nation (n : String) : Nation :=
{
  name := n,
  id := 1,
  gdp := BASE_GDP,                  -- example GDP in billions
  controlled_countries := 1,
  gdp_growth_rate := 0.02,
  economic_stability := 0.9,
  authoritarian_level := 0.3,
  regime_stability := 0.9,
  risk_tolerance := 0.2,
  globalism_level := 0.8,
  policy_inertia := 0.5,
  military_power := 200.0,
  military_growth_rate := 0.01,
  security_pressure := 0.2,
  war_losses := 0.0,
  war_fatigue := 0.0,
  resources := 100.0,
  claims := 0,
  supporters := 80,
  threat_perception_bias := 0.1,
  information_quality := 0.95,
  local_tension := 0.1,
  at_war := false,
  neighbors := 6,
  war_contribution := 0.0,
  gdp_max := 500.0 * 2,
  authoritarian_level_growth_rate := 0.3 / 100000,
  economic_stability_growth_rate := 0.00001,
  regime_stability_growth_rate := 0.01,
  risk_tolerance_growth_rate := 0.00001 * 0.5,
  war_fatigue_growth_rate := 0.0,
  globalism_level_growth_rate := 0.5 * 0.0001 * 0.8,
  global_losses := 0.0,

  x := 0.0,
  y := 0.0
}


-- if(this.regime_stability <= 0 || this.war_losses >= this.gdp+BASE_SURRENDER_THRESHOLD){
-- 


def distance (p1 : Nation) (p2 : Nation) : Float :=
  Float.sqrt (((p2.x - p1.x) ^ 2.0) + ((p2.y - p1.y) ^ 2.0))

def wartornNation : Nation :=
  { default_nation "Nation 1" with
      regime_stability := 0.0,
      war_losses := BASE_GDP + BASE_SURRENDER_THRESHOLD
  }

theorem surrenders : True := by
  let n1 := wartornNation
  have h : n1.regime_stability = 0.0 := rfl
  have h2 : n1.war_losses = BASE_GDP + BASE_SURRENDER_THRESHOLD := rfl
  trivial

--def attacker : Nation :=
--  { default_nation "Nation 2" with
--      regime_stability := 0.5,
 --     war_losses := BASE_GDP + BASE_SURRENDER_THRESHOLD
 -- }

--if (potentialTargets.length > 0 && this.regime_stability >0.5 && ((this.local_tension > (0.6-STUPIDITY_FLOOR) && this.globalism_level < 20) || (this.globalism_level >= 50 && this.local_tension>(0.9-STUPIDITY_FLOOR))) && this.at_war 
def attacker_neighbors : Nat := 1

axiom aggresive_war : True --unprovoked war has no criteria

--war is inevitable after prolonging peace
def n1 : Nation := {default_nation "N1" with local_tension := 0.0}
def n2 : Nation := {default_nation "N2" with local_tension := 0.0}

def hostility_factor : Rat := 0.01
def n1_neighbor: Nat := 1
def n2_neighbor: Nat := 1

def wages_war_bool (attacker: Nation) : Bool :=
  decide (attacker.regime_stability > (0.5 : Rat)) && decide (attacker.local_tension > (0.6 : Rat))

-- after war, the belligrents's aggressive level resets to 0 due to civilian pressure but the mean field models will not work if a "stupidity war after prolonged peace factor" is not introduced

--after some days, if the stupidity factor is not 0.0, a war will occur if the country is strong enough.
def f : ℚ := 1 / hostility_factor
def f_mul_hf : ℚ := f*hostility_factor
theorem wars_after_peace : True := by

  have h : n1_neighbor = 1 := rfl
  have h2 : n2_neighbor = 1 := rfl
  have h_factor_not_null : True := trivial

  -- By default they won't wage war. Just check it.
  have h3 : wages_war_bool n1 = false := by native_decide
  have h4 : wages_war_bool n2 = false := by native_decide

  apply aggresive_war

end

end CivSim