import Mathlib

structure Data where
  value  : String
  atomic : Bool
  depends_fully_on_primary_keys : Bool
  key : Bool
  transitive_dependency_from_primary_key : Bool
deriving Repr, Inhabited

def initData
  (value : String)
  (atomic : Bool)
  (depends_fully_on_primary_keys : Bool)
  (key : Bool)
  (transitive_dependency_from_primary_key : Bool)
: Data :=
  { value := value,atomic := atomic,depends_fully_on_primary_keys := depends_fully_on_primary_keys, key := key, transitive_dependency_from_primary_key := transitive_dependency_from_primary_key}


structure Record where
  entries : Array Data
deriving Repr, Inhabited

def initRecord : Record :=
  { entries := #[] }

def addToRecord (r : Record) (d : Data) : Record :=
  { r with entries := r.entries.push d }

def recordSize (r : Record) : Nat :=
  r.entries.size

def toString (r : Record) : String :=
  r.entries.foldl
    (fun acc d => acc ++ s!"[{d.value}, atomic={d.atomic}] ")
    ""

def atomicEntries (r : Record) : Array Data :=
  r.entries.filter (·.atomic) -- · is a free variable

-- 1NF if all data members are atomic
def one_NF (r : Record) : Prop :=
  ((atomicEntries r)).size = (recordSize r)
  -- or r.entries.all (·.atomic)

-- 0NF is the record if not 1NF
def zero_NF (r: Record) : Prop :=
  ¬ (one_NF r)

-- 2NF if 1NF and no second attribute depends partially on any first attribute (so all secondary attributes depend fully on all key attributes)
def two_NF (r: Record) : Prop :=
  (one_NF r) ∧ (r.entries.filter (fun d => d.depends_fully_on_primary_keys)).all (·.key)

-- 3NF: Egy séma akkor 3NF akkor ha 1NF és egyik másodlagos attributum sem függ tranzitívan a kulcstól.
def hasTransitiveDep (r : Record) : Prop :=
  (r.entries.filter (fun d => d.transitive_dependency_from_primary_key)).size ≠ 0

def three_NF (r: Record) : Prop :=
  (one_NF r) ∧ ( ¬ (hasTransitiveDep r) )

--BCNF: 1NF és attributum nem függ tranzitívan kulcstól.
def BCNF (r: Record) : Prop :=
  (one_NF r) ∧ ( ¬ (hasTransitiveDep r) )

--Examples
def r1 : Record :=
  initRecord
    |> (addToRecord · (initData "A" true true true false))
    |> (addToRecord · (initData "B" true true false false))

def r2 : Record :=
  initRecord
    |> (addToRecord · (initData "A" true true true false))
    |> (addToRecord · (initData "B" false true true false))