package hu.afghangoat.blockchain;

import java.awt.*;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.io.FileWriter;
import java.util.Scanner;

import java.util.Base64;

/**
 * @class BlockChainHelper
 * @brief An instantiable helper class which can load the encrypted blockchain files.
 *
 * Also validates the blockchain.
 *
 */
public class BlockChainHelper {

    /**
     * @brief The blocks of the chain will be stored here in an ordered manner.
     */
    private List<Block> blocks = new ArrayList<Block>();

    /**
     * @brief Initializes the blockchain with a default Genesis block.
     *
     * This will not be shown in the top list.
     *
     * @param initialHash The hash of the genesis block.
     */
    public BlockChainHelper(int initialHash){
        //Hash of prev block, genesis.
        addBlock("Genesis",-1,initialHash);

        /*addBlock("smth",8);
        if(verifyHash()==false){
            System.out.println("Error loading top list entries. Failed to verify blockchain hash integrity!");
        }*/

    }

    /**
     * @brief Verifies the inner blockchain by the previous hashes.
     *
     * @return whether the integrity of the blockchain is up.
     */
    public boolean verifyHash(){
        System.out.println("Hash size: "+blocks.size());
        for (int i = 0; i < blocks.size()-1; i++) {

            Block current = blocks.get(i);
            Block next = blocks.get(i + 1);

            //System.out.println("Generated: "+current.hashCode()+" | old:"+next.getPrevHash());

            if (current.hashCode() != next.getPrevHash()) {
                System.out.println("Hash verification failed at i: "+i);
                return false;
            } else {
                System.out.println(current.hashCode()+" | "+ next.getPrevHash());
            }
        }
        return true;
    }

    /**
     * @brief Clears all blocks except the Genesis block.
     *
     * This method assumes a genesis block exists.
     *
     */
    public void clearEntries() {
        if (blocks.size() > 1) { //The genesis block needs to stay
            blocks.subList(1, blocks.size()).clear();
        }
    }

    /**
     * @brief Adds a new block to the blockchain based on a single transaction which has a name and a score entry.
     *
     * @param nameEntry The name of the submitter.
     * @param score The score of the submitter.
     *
     */
    public void addBlock(String nameEntry, int score){

        Transaction cur = new Transaction(nameEntry,score);
        int prevHashCode=blocks.get(blocks.size()-1).hashCode();

        blocks.add(new Block(prevHashCode,(Transaction[]) new Transaction[]{cur}));
    }

    /**
     * @brief Adds a new block to the blockchain based on a single transaction which has a name and a score entry.
     *
     * Also takes in the previous hash into account.
     *
     * @param nameEntry The name of the submitter.
     * @param score The score of the submitter.
     * @param prevHashCode The hash of the previous block.
     *
     */
    public void addBlock(String nameEntry, int score, int prevHashCode){

        Transaction cur = new Transaction(nameEntry,score);

        blocks.add(new Block(prevHashCode,(Transaction[]) new Transaction[]{cur}));
    }

    /**
     * @brief Creates a HTML unordered list element from the blockchain.
     *
     * Excludes the Genesis block from the list.
     * It can be configured that only the N-th sublist from the chain should be returned.
     * The list will be ordered by the top scores in descending order.
     *
     * @param topEntryCount How much entries should be returned from the top list.
     *
     * @return The unordered HTML element.
     *
     */
    public String toHTMLSorted(int topEntryCount){
        String value="<html><ul>";

        List<Block> sortedData = new ArrayList<Block>();
        for(int i=1;i<blocks.size();i++){ //Skip genesis block, displaying that is of no importance.
            sortedData.add(blocks.get(i).clone());
        }

        Collections.sort(sortedData,new BlockComparator());
        Collections.reverse(sortedData);

        for(int i=0;i<Math.min(sortedData.size(),topEntryCount);i++){
            Transaction cur=sortedData.get(i).getTransactions()[0];
            value += "<li><b> #" +(i+1)+"</b>"+cur.getSource()+" - "+cur.getScore()+"</li>";
        }

        value+="</ul></html>";

        return value;
    }

    /**
     * @brief Adds a new block to the blockchain based on multiple transactions which has a name and a score entry.
     *
     * @param transactions A list of transactions.
     * @param hash The hash of the block.
     *
     */
    public void addBlock(Transaction[] transactions,int hash){

        //int prevHashCode=blocks.get(blocks.size()-1).hashCode();

        blocks.add(new Block(hash,transactions));
    }

    /**
     * @brief Converts a string to a transaction.
     *
     * @param data The input string.
     * @return the converted transaction.
     *
     */
    public Transaction transactionFromString(String data){
        //Source, score 1,3
        String[] entries = data.split(":");

        return new Transaction(entries[1],Integer.parseInt(entries[3]));

    }

    /**
     * @brief Saves the stored blockchain to a supplied filename to the config directory.
     *
     * Includes the hashes as well as it applies a base64 encoding for a little more security.
     * @implNote The real security lies withing the blockchain integrity.
     * If the integrity is not active, the savefile will be marked as "cheated".
     *
     * @param fileName The name of the savefile
     * @return Whether the IO was successful.
     *
     */
    public boolean saveToFile(String fileName){
        try {
            File saveFile = new File(fileName);
            if (saveFile.createNewFile()) {
                System.out.println("File created: " + saveFile.getName());
            } else {
                System.out.println("File already exists, replacing it...");
                //return false;
            }
        } catch (IOException e) {
            System.out.println("An error occurred.");
            e.printStackTrace();
        }

        try {
            FileWriter writer = new FileWriter(fileName);

            String line_entry="";
            for(int i=1;i<blocks.size();i++){

                int transactionSize=blocks.get(i).getTransactions().length;


                //writer.write(Integer.toString(blocks.get(i).getPrevHash())+'|'+transactionSize+'|');
                line_entry+=Integer.toString(blocks.get(i).getPrevHash())+'|'+transactionSize+'|';

                for(int j=0;j<transactionSize;j++) {
                    Transaction cur = blocks.get(i).getTransactions()[j];
                    //writer.write(cur.toString() + '|');
                    line_entry+=cur.toString() + '|';
                }

                //line_entry+='\n';

            }
            byte[] bytes = line_entry.getBytes("UTF-8");
            String encoded = Base64.getEncoder().encodeToString(bytes);

            writer.write(encoded);

            writer.close();
            System.out.println("Successfully wrote to the file.");
        } catch (IOException e) {
            System.out.println("An error occurred.");
            e.printStackTrace();
        }

        return true;
    }

    /**
     * @brief Loads and decodes a blockchain store file which stores the top list entries.
     *
     * @param fileName The name of the savefile.
     * @return Whether the IO was successful.
     *
     */
    public boolean loadFromFile(String fileName){
        try {
            File fileForRead = new File(fileName);


            if (!fileForRead.exists()) {
                System.out.println("File does not exist: " + fileForRead.getAbsolutePath()+"! Creating empty save file.");
                return saveToFile(fileName);
            }

            Scanner myReader = new Scanner(fileForRead);

            clearEntries();

            while (myReader.hasNextLine()) {

                byte[] decoded = Base64.getDecoder().decode(myReader.nextLine());
                String cur_line = new String(decoded, StandardCharsets.UTF_8);
                String[] entries = cur_line.split("\\|");

                //System.out.println("Entries size:"+entries.length);


                int transactionSize = entries.length;

                if(transactionSize==0){
                    System.out.println("Invalid/Corrupted scoreboard save file! Skipping...");
                    return false;
                }

                //
                for(int i=0;i<transactionSize/3;i++) {
                    int hash = Integer.parseInt(entries[i * 3]);
                    Transaction[] transactionsInLine = new Transaction[1];
                    transactionsInLine[0] = transactionFromString(entries[2 + i * 3]);
                    //System.out.println("Added T"+i+": "+transactionsInLine[0]);
                    addBlock(transactionsInLine, hash);
                }


            }

            myReader.close();
            return true;
        } catch (FileNotFoundException e) {

            System.out.println("An error occurred.");
            e.printStackTrace();
        }

        return false;
    }
}
