package hu.afghangoat.helpers;

/**
 * @class GPSCoordinate
 * @brief A data structure which holds GPS longitude and latitude pairs.
 *
 */
public class GPSCoordinate {

    /**
     * @brief The GPS latitude of the instance data structure pair.
     */
    double GPSLatitude = 0.0;

    /**
     * @brief The GPS longitude of the instance data structure pair.
     */
    double GPSLongitude = 0.0;

    /**
     * @brief The GPS origo latitude coordinate. This is the center on my map.
     */
    public static final double BASE_LATITUDE=47.473622222222225;

    /**
     * @brief The GPS origo longitude coordinate. This is the center on my map.
     */
    public static final double BASE_LONGITUDE=19.059227777777778;

    /**
     * @brief The offset on the X axis in pixels. This will be used in flatmap conversion.
     */
    public static final double FLATMAP_OFFSET_X=2348.7726148877664;

    /**
     * @brief The offset on the Y axis in pixels. This will be used in flatmap conversion.
     */
    public static final double FLATMAP_OFFSET_Y=3247.3495229108244;

    /**
     * @brief This will be used in calculating the latitude formula.
     */
    public static final double EARTH_TO_PIXEL_PARAMETER=156543.03392;

    /**
     * @brief The conversion scale based on the parameters of the earth.
     *
     * Let's say that someone wants to make a UNIGuesser franchise on an asteroid which is not round.
     * That also needs to be addressed.
     */
    public static final double CONVERSION_SCALE = EARTH_TO_PIXEL_PARAMETER*Math.PI;

    /**
     * @brief Whether the GPS coordinates are on the northern or on the southern hemisphere.
     */
    public static final boolean IS_ON_NORTHERN_HEMISPHERE=true;

    /**
     * @brief The conversion scale based on the parameters of the earth.
     *
     * Let's say that someone wants to make a UNIGuesser franchise on an asteroid which is not round.
     * That also needs to be addressed.
     */
    public static final int ZOOM = 18;

    /**
     * @brief The rotation modifier which will determine whether the Y and X cartesian axes will align up or down.
     */
    public static final double rotationModifier=decideRotationModifier();

    /**
     * @brief Calculates the rotation modifier based on the northern or southern hemisphere.
     *
     * @return The rotational modifier.
     */
    public static double decideRotationModifier() {
        double r=1.0;
        if(IS_ON_NORTHERN_HEMISPHERE==true){
            r=-1.0;
        }
        return r;
    }

    /**
     * @brief Gets the pixels per meter ratio based on how zoomed the user is and how close the coordinate is to the northern point.
     *
     * @param zoom How zoomed the user is. Uses OpenStreetMaps zoom levels.
     *
     * @param baseLat The latitude of the GPS coordinate.
     *
     * @return The pixels/meter ratio of the schema.
     */
    public static double getPixelsPerMeter(int zoom, double baseLat){
        return (EARTH_TO_PIXEL_PARAMETER*Math.cos(Math.toRadians(baseLat)))/Math.pow(2,zoom);
    }

    /**
     * @brief Stores the pixels/meter ratio to the current schema.
     */
    public static double pixelsPerMeter=getPixelsPerMeter(ZOOM,BASE_LATITUDE);

    /**
     * @brief A getter for the latitude.
     *
     * @return The GPS coordinates latitude.
     */
    public double getGPSLatitude() {
        return GPSLatitude;
    }


    /**
     * @brief A getter for the longitude.
     *
     * @return The GPS coordinates longitude.
     */
    public double getGPSLongitude() {
        return GPSLongitude;
    }

    /**
     * @brief A basic constructor which makes a coordinate from an input longitude and latitude.
     *
     * @param lat The latitude of the coordinate.
     * @param lon The longitude of the coordinate.
     */
    public GPSCoordinate(double lat, double lon) {
        this.GPSLatitude = lat;
        this.GPSLongitude = lon;
    }

    /**
     * @brief Converts the GPS coordinate to a flatmap cartesian coordinate.
     *
     * @return The converted flatmap coordinate.
     */
    public FlatMapCoordinate toFlatMapCoordinate(){
        //System.out.println("Converted to: "+((BASE_LATITUDE-this.GPSLatitude)*CONVERSION_SCALE*pixelsPerMeter)+" | "+((BASE_LONGITUDE-this.GPSLongitude)*CONVERSION_SCALE*pixelsPerMeter));

        return new FlatMapCoordinate(FLATMAP_OFFSET_X-(BASE_LONGITUDE-this.GPSLongitude)*CONVERSION_SCALE*pixelsPerMeter*rotationModifier, FLATMAP_OFFSET_Y-(BASE_LATITUDE-this.GPSLatitude)*CONVERSION_SCALE*pixelsPerMeter*rotationModifier);
    }

    /**
     * @brief Converts a given flatmap cartesian coordinate to a GPS coordinate.
     *
     * @param coord The input flatmap coordinate.
     *
     * @return The converted GPS coordinate.
     */
    public static GPSCoordinate fromFlatMapCoordinate(FlatMapCoordinate coord){
        /*
        * px = offset-(base-x)*scale
        * (px-offset)/scale = -base + x
        * (px-offset)/scale + base = x
        *
        *
        *
        * */

        return new GPSCoordinate((coord.getPosY()-FLATMAP_OFFSET_Y)/(CONVERSION_SCALE*pixelsPerMeter*rotationModifier) + BASE_LATITUDE,(coord.getPosX()-FLATMAP_OFFSET_X)/(CONVERSION_SCALE*pixelsPerMeter*rotationModifier) + BASE_LONGITUDE);
    }
}
