package hu.afghangoat.helpers;

import java.io.*;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.file.Files;
import java.nio.file.Path;
//import com.drew.imaging.ImageMetadataReader;
//import com.drew.metadata.Directory;
//import com.drew.metadata.Metadata;
//import com.drew.metadata.Tag;

/**
 * @class GPSReader
 * @brief A utility class which is able to extract GPS coordinates from JPEG images.
 *
 * Not used by default.
 *
 */
public class GPSReader {

    /**
     * @brief The GPS info tag byte. Here starts the information stream of the GPS data.
     */
    public static final int GPS_INFO_TAG = 0x8825;

    /**
     * @brief The APP1 marker is needed for finding the GPS latitude and longitude coordinate.
     */
    public static final int APP1_MARKER = 0xE1;

    /**
     * @brief The start segment byte of the coordinate first marker.
     */
    public static final int MARKER_START_SEGMENT = 0xFF;

    /**
     * @brief The JPEG start-of-image byte's first part.
     *
     * Needed for verification.
     */
    public static final int SOI_START_BYTE1 = 0xFF;

    /**
     * @brief The JPEG start-of-image byte's second part.
     *
     * Needed for verification.
     */
    public static final int SOI_START_BYTE2 = 0xD8;

    /**
     * @brief Empty default constructor.
     */
    public GPSReader(){

    }

    /**
     * @brief This method takes in a file and tries to extract the GPS coordinates from it.
     *
     * It is important that the method will throw an IO exception if the file is unreachable or the file is not a JPEG file.
     *
     * @param file The input file.
     *
     * @return The extracted GPS coordinate in success. Otherwise, null or error.
     */
    public GPSCoordinate extractGps(File file) throws IOException {
        try (FileInputStream fis = new FileInputStream(file)) {
            //Check JPEG SOI (Start of image)
            if (fis.read() != SOI_START_BYTE1 || fis.read() != SOI_START_BYTE2) {
                throw new IOException("Not a JPEG file");
            }

            //Iterate over segments
            while (true) {
                int marker1 = fis.read();
                int marker2 = fis.read();
                if (marker1 != MARKER_START_SEGMENT) break;
                int marker = marker2;

                int lenHi = fis.read();
                int lenLo = fis.read();
                int length = ((lenHi) << 8) | (lenLo);

                if (isApp1(marker)==true) { // APP1 (Exif)
                    byte[] data = new byte[length - 2];
                    fis.read(data);
                    return parseExifForGps(data);
                } else {
                    long skipping = fis.skip(length - 2);
                }
            }
        }
        return null;
    }

    /**
     * @brief Checks if a found marker is equal to the APP1 marker.
     *
     * @param marker the byte signal of the found marker.
     *
     * @return Whether the marker is an APP1 marker.
     */
    private boolean isApp1(int marker){
        return marker == APP1_MARKER;
    }

    /**
     * @brief This method tries to extract the GPS longitude and latitude from the EXIF info.
     *
     * If they do not exist in the EXIF data, null will be returned.
     *
     * @param data The byte stream of the EXIF data.
     *
     * @return The extracted GPS coordinate if found. Otherwise, null.
     */
    private GPSCoordinate parseExifForGps(byte[] data) throws IOException {
        // must start with "Exif\0\0"
        String header = new String(data, 0, 4, "ASCII");
        if (!header.equals("Exif")){
            return null;
        }

        // TIFF header starts at offset 6
        int tiffStart = 6;
        boolean gotByteOrder = (data[tiffStart] == 'I' && data[tiffStart + 1] == 'I');
        ByteOrder bo = gotByteOrder ? ByteOrder.LITTLE_ENDIAN : ByteOrder.BIG_ENDIAN;

        ByteBuffer buf = ByteBuffer.wrap(data);
        buf.order(bo);
        buf.position(tiffStart + 4);
        int ifd0Offset = buf.getInt();

        // parse IFD0 and look for tag 0x8825 (GPSInfo)
        int gpsOffset = findTagOffset(data, tiffStart + ifd0Offset, GPS_INFO_TAG, bo, tiffStart);
        if (gpsOffset == -1){
            return null;
        }

        // Now parse GPS IFD for tags 0x0001..0x0004
        String latRef = readAsciiTag(data, gpsOffset, 0x0001, bo, tiffStart);
        double lat = readRationalTag(data, gpsOffset, 0x0002, bo, tiffStart);
        String lonRef = readAsciiTag(data, gpsOffset, 0x0003, bo, tiffStart);
        double lon = readRationalTag(data, gpsOffset, 0x0004, bo, tiffStart);

        //String altRef = readAsciiTag(data, gpsOffset, 0x0005, bo, tiffStart);
        //double alt = readRationalTag(data, gpsOffset, 0x0006, bo, tiffStart);

        //Check for global directions

        if (latRef != null && latRef.equals("S")){
            lat = -lat;
        }

        if (lonRef != null && lonRef.equals("W")){
            lon = -lon;
        }

        //System.out.println("altRef: "+altRef+" alt: "+alt);

        return new GPSCoordinate(lat, lon);
    }

    /**
     * @brief Finds the byte offset for the GPS tag.
     *
     * Takes the byte order into account.
     * If the searched tag if not found, returns a -1.
     *
     * @param data The bit stream where the data needs to be found.
     * @param ifdOffset The offset of the IFD0 tag. Here are the GPS coordinates we need.
     * @param wantedTag The byte sign of the wanted tag we search.
     * @param bo The byte order.
     * @param tiffStart TIFF header ALWAYS starts at offset 6 but make sure, we are backwards and forwards compatible.
     *
     * @return The tag offset of the wanted tag from the beginning of the bit stream. -1 if the tag is not present.
     */
    private int findTagOffset(byte[] data, int ifdOffset, int wantedTag, ByteOrder bo, int tiffStart) {
        ByteBuffer buf = ByteBuffer.wrap(data);
        buf.order(bo);
        buf.position(ifdOffset);
        int numEntries = buf.getShort() & 0xFFFF;
        for (int i = 0; i < numEntries; i++) {
            int tag = buf.getShort() & 0xFFFF;
            int type = buf.getShort() & 0xFFFF;
            int count = buf.getInt();
            int valueOffset = buf.getInt();
            if (tag == wantedTag) {
                return tiffStart + valueOffset;
            }
        }
        return -1;
    }

    /**
     * @brief Reads and parses a wanted tag into an ascii text.
     *
     * Takes the byte order into account.
     * If the searched tag if not found, returns null.
     *
     * @param data The bit stream where the data needs to be found.
     * @param ifdOffset The offset of the IFD0 tag. Here are the GPS coordinates we need.
     * @param wantedTag The byte sign of the wanted tag we search.
     * @param bo The byte order.
     * @param tiffStart TIFF header ALWAYS starts at offset 6 but make sure, we are backwards and forwards compatible.
     *
     * @return The String representation of the searched tag. If not found, null.
     */
    private String readAsciiTag(byte[] data, int ifdOffset, int wantedTag, ByteOrder bo, int tiffStart) {
        ByteBuffer buf = ByteBuffer.wrap(data);
        buf.order(bo);
        buf.position(ifdOffset);
        int numEntries = buf.getShort() & 0xFFFF;
        for (int i = 0; i < numEntries; i++) {
            int tag = buf.getShort() & 0xFFFF;
            int type = buf.getShort() & 0xFFFF;
            int count = buf.getInt();
            int valueOffset = buf.getInt();
            if (tag == wantedTag) {
                int realOffset = (count <= 4) ? buf.position() - 4 : tiffStart + valueOffset;
                return new String(data, realOffset, count - 1); // ignore null terminator
            }
        }
        return null;
    }

    /**
     * @brief Can be used to read rational (float) tags from bytes.
     *
     * Will be used for GPS coordinate parsing.
     * If the tag is not found or can not be parsed, it returns 0.0.
     *
     * @param data The bit stream where the data needs to be found.
     * @param ifdOffset The offset of the IFD0 tag. Here are the GPS coordinates we need.
     * @param wantedTag The byte sign of the wanted tag we search.
     * @param bo The byte order.
     * @param tiffStart TIFF header ALWAYS starts at offset 6 but make sure, we are backwards and forwards compatible.
     *
     * @return The double representation of the tag if it is present. Otherwise, 0.0.
     */
    private double readRationalTag(byte[] data, int ifdOffset, int wantedTag, ByteOrder bo, int tiffStart) {
        ByteBuffer buf = ByteBuffer.wrap(data);
        buf.order(bo);
        buf.position(ifdOffset);
        int numEntries = buf.getShort() & 0xFFFF;
        for (int i = 0; i < numEntries; i++) {
            int tag = buf.getShort() & 0xFFFF;
            int type = buf.getShort() & 0xFFFF;
            int count = buf.getInt();
            int valueOffset = buf.getInt();
            if (tag == wantedTag) {
                int realOffset = tiffStart + valueOffset;
                double[] vals = new double[count];
                for (int j = 0; j < count; j++) {
                    int num = toInt(data, realOffset + j * 8, bo);
                    int den = toInt(data, realOffset + j * 8 + 4, bo);
                    vals[j] = den == 0 ? 0 : ((double) num / den);
                }
                // degrees + minutes/60 + seconds/3600
                return vals[0] + vals[1] / 60.0 + vals[2] / 3600.0;
            }
        }
        return 0.0;
    }

    /**
     * @brief Parses a part of the bit stream to an int, taking an offset and a byte order into account.
     *
     * If the tag can not be parsed, it returns a -1 integer value.
     *
     * @param data The bit stream where the data needs to be found.
     * @param offset The offset of the searched tag.
     * @param bo The byte order.
     *
     * @return The parsed integer or if the parsing was invalid, -1 to be returned.
     */
    private int toInt(byte[] data, int offset, ByteOrder bo) {
        //Shift bytes to correct position.
        if (bo == ByteOrder.LITTLE_ENDIAN) { //Or "II"
            return (data[offset] & 0xFF) | // byte 0 -> bits 0–7
                    ((data[offset + 1] & 0xFF) << 8) | // byte 1 -> bits 8–15
                    ((data[offset + 2] & 0xFF) << 16) | // byte 2 -> bits 16–23
                    ((data[offset + 3] & 0xFF) << 24); // byte 3 -> bits 24–31
        } else if (bo == ByteOrder.BIG_ENDIAN){ //Or "MM"
            return ((data[offset] & 0xFF) << 24) | // byte 0 -> bits 24–31
                    ((data[offset + 1] & 0xFF) << 16) | // byte 1 -> bits 16–23
                    ((data[offset + 2] & 0xFF) << 8) | // byte 2 -> bits 8–15
                    (data[offset + 3] & 0xFF); // byte 3 -> bits 0–7
        }
        System.out.println("Invalid byteorder, skipping GPSReader.toInt() call!");
        return -1;
    }

    /*public void printEXIFData(String imagePath) throws Exception {
        File jpegFile = new File(imagePath);
        Metadata metadata = ImageMetadataReader.readMetadata(jpegFile);

        for (Directory directory : metadata.getDirectories()) {
            for (Tag tag : directory.getTags()) {
                System.out.println(tag);
            }
        }
    }*/

}

