package hu.afghangoat.views;

import hu.afghangoat.LangParser;
import hu.afghangoat.widgets.BackButton;
import hu.afghangoat.widgets.FancyButton;
import hu.afghangoat.widgets.GradientPanel;

import javax.swing.*;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

/**
 * @class EndGamePanel
 * @brief The panel which shows the final game UI.
 *
 * Here, if the user was playing cleverly, can input the name which will be used at the leaderboard representation.
 *
 */
public class EndGamePanel extends GradientPanel implements ViewInterface, ActionListener {

    /**
     * @brief This is the de-facto, hardcoded length of a leaderboard name.
     */
    public static final int MAX_NAME_LENGTH=20;

    /**
     * @brief Stores the end game title text language-wise.
     */
    private JLabel label = new JLabel("END_GAME_TITLE");

    /**
     * @brief Clicking the "back" button will take the user back to the main menu.
     */
    private BackButton backBtn;

    /**
     * @brief Submitting the score will save the entry to the top list file.
     */
    private JButton submitButton;

    /**
     * @brief The user can input its name here in the text area.
     */
    private JTextArea nameArea = new JTextArea("EDIT_NAME");

    /**
     * @brief The final score of the user stored here.
     */
    private int finalScore=0;

    /**
     * @brief This method handles what should happen when the language is changed.
     */
    @Override
    public void langChanged(){
        label.setText(LangParser.getLang("END_GAME_TITLE"));
        backBtn.setText(LangParser.getLang("BACK"));
        submitButton.setText(LangParser.getLang("SUBMIT"));
        nameArea.setText(LangParser.getLang("EDIT_NAME"));
    }

    /**
     * @brief This method appends the score to the body layout of the panel language-wise.
     *
     * @param score The score which needs to be shown.
     */
    public void showScore(int score){
        finalScore=score;
        langChanged();

        //System.out.println(LangParser.getLang("END_GAME_TITLE")+": "+finalScore);
        label.setText((LangParser.getLang("END_GAME_TITLE")+": ").replace("$",Integer.toString(finalScore)));
        label.revalidate();
        label.repaint();
    }

    /**
     * @brief This constructor sets up the layout of the panel and sets the event listeners. Also takes the parent window account.
     *
     * @param mainWindow The parent window of the panel.
     */
    public EndGamePanel(MainWindow mainWindow) {
        super(new GridBagLayout());

        //setLayout();

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = GridBagConstraints.RELATIVE;
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.fill = GridBagConstraints.NONE;
        //label_gbc.weightx = 1.0;
        gbc.insets = new Insets(0, 10, 25, 10); // spacing

        backBtn = new BackButton(mainWindow,"BACK");

        label.setFont(new Font("Segoe UI", Font.BOLD, 30));
        add(label,gbc);

        nameArea.setMargin(new Insets(10, 10, 10, 10));
        nameArea.setMinimumSize(new Dimension(200, 28));

        nameArea.getDocument().addDocumentListener(new DocumentListener() {
            private void check() {
                if (nameArea.getText().length() > MAX_NAME_LENGTH) {
                    SwingUtilities.invokeLater(() -> {
                        nameArea.setText(nameArea.getText().substring(0, MAX_NAME_LENGTH));
                    });
                }
                //1876207995|1|Transaction{source=':Hehe:', score=:99:}|
                if(nameArea.getText().contains(":")){
                    SwingUtilities.invokeLater(() -> {
                        nameArea.setText(nameArea.getText().replaceAll(":", ""));
                    });
                }
                if(nameArea.getText().contains("|")){
                    SwingUtilities.invokeLater(() -> {
                        nameArea.setText(nameArea.getText().replaceAll("\\|", ""));
                    });
                }
                if(nameArea.getText().contains("'")){
                    SwingUtilities.invokeLater(() -> {
                        nameArea.setText(nameArea.getText().replaceAll("'", ""));
                    });
                }
            }
            public void insertUpdate(DocumentEvent e) { check(); }
            public void removeUpdate(DocumentEvent e) { }
            public void changedUpdate(DocumentEvent e) { }
        });

        add(nameArea, gbc);

        submitButton = new FancyButton("SUBMIT");

        add(submitButton, gbc);

        add(backBtn,gbc);

        submitButton.addActionListener(e -> {

            String newName = nameArea.getText();
            mainWindow.addTopListEntry(newName,finalScore);
            mainWindow.showView("MENU");

        });
    }

    /**
     * @brief Empty action listener, needs to be implemented.
     *
     * @param e The event of the action.
     * @deprecated Moved logic to the constructor.
     */
    @Override
    @Deprecated
    public void actionPerformed(ActionEvent e) {

    }
}