package hu.afghangoat.widgets;

import javax.imageio.ImageIO;
import javax.swing.*;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.awt.image.ImageObserver;
import java.io.File;
import java.io.IOException;

/**
 * @class ScrollableImage
 * @brief A scrollable image which can be used to render a panorama image.
 *
 * Supports left-right scrolling and panning.
 */
public class ScrollableImage extends ImageIcon {

    /**
     * @brief The maximum width of the image.
     */
    private int maxLengthX=0;

    /**
     * @brief The frame-fit target width of the image.
     */
    private int itargetWidth=0;

    /**
     * @brief The frame-fit target height of the image.
     */
    private int itargetHeight=0;

    /**
     * @brief The current X offset from the left border of the image.
     */
    private int currentPosX=0;

    /**
     * @brief The frame-fit image segment.
     */
    private BufferedImage output;

    /**
     * @brief The frame-fit scaled image segment.
     */
    private BufferedImage scaledBuffered;

    /**
     * @brief A reference to the original image.
     */
    private BufferedImage original;

    /**
     * @brief The frame-fit new width of the image after the scaling.
     */
    private int newWidth;

    /**
     * @brief The frame-fit new height of the image after the scaling.
     */
    private int newHeight;

    /**
     * @brief A basic constructor calling the Swing ImageIcon's constructor.
     */
    public ScrollableImage(){
        super();
    }

    /**
     * @brief This constructor sets the scrollable image to a scaled-down, usable target image.
     *
     * @param path The path of the image.
     * @param targetWidth the scaled-down width of the image.
     * @param targetHeight the scaled-down height of the image.
     */
    public ScrollableImage(String path, int targetWidth, int targetHeight){
        itargetWidth= targetWidth;
        itargetHeight=targetHeight;
        currentPosX=0;

        try {
            original = ImageIO.read(new File(path));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        int originalWidth = original.getWidth();
        int originalHeight = original.getHeight();

        //double scaleX = (double) targetWidth / originalWidth;
        double scaleY = (double) targetHeight / originalHeight;

        double scale = scaleY;//Math.max(scaleX, scaleY);

        newWidth = (int) (originalWidth * scale);
        newHeight = (int) (originalHeight * scale);

        Image temp = original.getScaledInstance(newWidth, newHeight, Image.SCALE_SMOOTH);
        scaledBuffered = new BufferedImage(newWidth, newHeight, BufferedImage.TYPE_INT_ARGB);
        Graphics2D g = scaledBuffered.createGraphics();
        g.drawImage(temp, 0, 0, null);
        g.dispose();

        maxLengthX=Math.max(0,newWidth-targetWidth);

        moveViewport(currentPosX);

    }

    /**
     * @brief Sets a new offset from the left side of the image.
     *
     * @param posX The new offset.
     */
    public void moveViewport(int posX) {
        currentPosX += posX;
        int safePosX = Math.max(0, Math.min(currentPosX, maxLengthX));
        currentPosX = safePosX;

        output = new BufferedImage(itargetWidth, itargetHeight, BufferedImage.TYPE_INT_ARGB);

        Graphics2D g2 = output.createGraphics();
        g2.setComposite(AlphaComposite.Clear);
        g2.fillRect(0, 0, output.getWidth(), output.getHeight());
        g2.setComposite(AlphaComposite.SrcOver);

        int srcX1 = safePosX;
        int srcY1 = 0;
        int srcX2 = Math.min(safePosX + output.getWidth(), scaledBuffered.getWidth());
        int srcY2 = scaledBuffered.getHeight();

        int destWidth = srcX2 - srcX1;
        int destHeight = srcY2 - srcY1;
        int destY = (output.getHeight() - destHeight) / 2;

        g2.drawImage(scaledBuffered,
                0, destY, destWidth, destY + destHeight,
                srcX1, srcY1, srcX2, srcY2,
                null);

        g2.dispose();
        setImage(output);
    }

    /**
     * @brief Sets a new panorama image.
     *
     * @param img The new image.
     */
    private void setImage(BufferedImage img){
        this.output = img;
        super.setImage(img); // update the ImageIcon's internal image
    }


}
